"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.isNestedMetaUpdate = exports.applySnapshot = void 0;
const lodash_1 = require("lodash");
const database_1 = __importDefault(require("../database"));
const logger_1 = __importDefault(require("../logger"));
const services_1 = require("../services");
const get_schema_1 = require("./get-schema");
const get_snapshot_1 = require("./get-snapshot");
const get_snapshot_diff_1 = require("./get-snapshot-diff");
const cache_1 = require("../cache");
async function applySnapshot(snapshot, options) {
    var _a, _b, _c, _d;
    const database = (_a = options === null || options === void 0 ? void 0 : options.database) !== null && _a !== void 0 ? _a : (0, database_1.default)();
    const schema = (_b = options === null || options === void 0 ? void 0 : options.schema) !== null && _b !== void 0 ? _b : (await (0, get_schema_1.getSchema)({ database }));
    const { systemCache } = (0, cache_1.getCache)();
    const current = (_c = options === null || options === void 0 ? void 0 : options.current) !== null && _c !== void 0 ? _c : (await (0, get_snapshot_1.getSnapshot)({ database, schema }));
    const snapshotDiff = (_d = options === null || options === void 0 ? void 0 : options.diff) !== null && _d !== void 0 ? _d : (0, get_snapshot_diff_1.getSnapshotDiff)(current, snapshot);
    await database.transaction(async (trx) => {
        const collectionsService = new services_1.CollectionsService({ knex: trx, schema });
        const getNestedCollectionsToCreate = (currentLevelCollection) => snapshotDiff.collections.filter(({ diff }) => { var _a, _b; return ((_b = (_a = diff[0].rhs) === null || _a === void 0 ? void 0 : _a.meta) === null || _b === void 0 ? void 0 : _b.group) === currentLevelCollection; });
        const getNestedCollectionsToDelete = (currentLevelCollection) => snapshotDiff.collections.filter(({ diff }) => { var _a, _b; return ((_b = (_a = diff[0].lhs) === null || _a === void 0 ? void 0 : _a.meta) === null || _b === void 0 ? void 0 : _b.group) === currentLevelCollection; });
        const createCollections = async (collections) => {
            for (const { collection, diff } of collections) {
                if ((diff === null || diff === void 0 ? void 0 : diff[0].kind) === 'N' && diff[0].rhs) {
                    // We'll nest the to-be-created fields in the same collection creation, to prevent
                    // creating a collection without a primary key
                    const fields = snapshotDiff.fields
                        .filter((fieldDiff) => fieldDiff.collection === collection)
                        .map((fieldDiff) => fieldDiff.diff[0].rhs)
                        .map((fieldDiff) => {
                        var _a, _b, _c, _d, _e;
                        // Casts field type to UUID when applying non-PostgreSQL schema onto PostgreSQL database.
                        // This is needed because they snapshots UUID fields as char with length 36.
                        if (((_a = fieldDiff.schema) === null || _a === void 0 ? void 0 : _a.data_type) === 'char' &&
                            ((_b = fieldDiff.schema) === null || _b === void 0 ? void 0 : _b.max_length) === 36 &&
                            (((_c = fieldDiff.schema) === null || _c === void 0 ? void 0 : _c.is_primary_key) ||
                                (((_d = fieldDiff.schema) === null || _d === void 0 ? void 0 : _d.foreign_key_table) && ((_e = fieldDiff.schema) === null || _e === void 0 ? void 0 : _e.foreign_key_column)))) {
                            return (0, lodash_1.merge)(fieldDiff, { type: 'uuid', schema: { data_type: 'uuid', max_length: null } });
                        }
                        else {
                            return fieldDiff;
                        }
                    });
                    try {
                        await collectionsService.createOne({
                            ...diff[0].rhs,
                            fields,
                        });
                    }
                    catch (err) {
                        logger_1.default.error(`Failed to create collection "${collection}"`);
                        throw err;
                    }
                    // Now that the fields are in for this collection, we can strip them from the field edits
                    snapshotDiff.fields = snapshotDiff.fields.filter((fieldDiff) => fieldDiff.collection !== collection);
                    await createCollections(getNestedCollectionsToCreate(collection));
                }
            }
        };
        const deleteCollections = async (collections) => {
            for (const { collection, diff } of collections) {
                if ((diff === null || diff === void 0 ? void 0 : diff[0].kind) === 'D') {
                    const relations = schema.relations.filter((r) => r.related_collection === collection || r.collection === collection);
                    if (relations.length > 0) {
                        const relationsService = new services_1.RelationsService({ knex: trx, schema });
                        for (const relation of relations) {
                            try {
                                await relationsService.deleteOne(relation.collection, relation.field);
                            }
                            catch (err) {
                                logger_1.default.error(`Failed to delete collection "${collection}" due to relation "${relation.collection}.${relation.field}"`);
                                throw err;
                            }
                        }
                        // clean up deleted relations from existing schema
                        schema.relations = schema.relations.filter((r) => r.related_collection !== collection && r.collection !== collection);
                    }
                    await deleteCollections(getNestedCollectionsToDelete(collection));
                    try {
                        await collectionsService.deleteOne(collection);
                    }
                    catch (err) {
                        logger_1.default.error(`Failed to delete collection "${collection}"`);
                        throw err;
                    }
                }
            }
        };
        // Finds all collections that need to be created
        const filterCollectionsForCreation = ({ diff }) => {
            var _a;
            // Check new collections only
            const isNewCollection = diff[0].kind === 'N';
            if (!isNewCollection)
                return false;
            // Create now if no group
            const groupName = (_a = diff[0].rhs.meta) === null || _a === void 0 ? void 0 : _a.group;
            if (!groupName)
                return true;
            // Check if parent collection already exists in schema
            const parentExists = current.collections.find((c) => c.collection === groupName) !== undefined;
            // If this is a new collection and the parent collection doesn't exist in current schema ->
            // Check if the parent collection will be created as part of applying this snapshot ->
            // If yes -> this collection will be created recursively
            // If not -> create now
            // (ex.)
            // TopLevelCollection - I exist in current schema
            // 		NestedCollection - I exist in snapshotDiff as a new collection
            //			TheCurrentCollectionInIteration - I exist in snapshotDiff as a new collection but will be created as part of NestedCollection
            const parentWillBeCreatedInThisApply = snapshotDiff.collections.filter(({ collection, diff }) => diff[0].kind === 'N' && collection === groupName)
                .length > 0;
            // Has group, but parent is not new, parent is also not being created in this snapshot apply
            if (parentExists && !parentWillBeCreatedInThisApply)
                return true;
            return false;
        };
        // Create top level collections (no group, or highest level in existing group) first,
        // then continue with nested collections recursively
        await createCollections(snapshotDiff.collections.filter(filterCollectionsForCreation));
        // delete top level collections (no group) first, then continue with nested collections recursively
        await deleteCollections(snapshotDiff.collections.filter(({ diff }) => { var _a; return diff[0].kind === 'D' && ((_a = diff[0].lhs.meta) === null || _a === void 0 ? void 0 : _a.group) === null; }));
        for (const { collection, diff } of snapshotDiff.collections) {
            if ((diff === null || diff === void 0 ? void 0 : diff[0].kind) === 'E' || (diff === null || diff === void 0 ? void 0 : diff[0].kind) === 'A') {
                const newValues = snapshot.collections.find((field) => {
                    return field.collection === collection;
                });
                if (newValues) {
                    try {
                        await collectionsService.updateOne(collection, newValues);
                    }
                    catch (err) {
                        logger_1.default.error(`Failed to update collection "${collection}"`);
                        throw err;
                    }
                }
            }
        }
        const fieldsService = new services_1.FieldsService({ knex: trx, schema: await (0, get_schema_1.getSchema)({ database: trx }) });
        for (const { collection, field, diff } of snapshotDiff.fields) {
            if ((diff === null || diff === void 0 ? void 0 : diff[0].kind) === 'N' && !isNestedMetaUpdate(diff === null || diff === void 0 ? void 0 : diff[0])) {
                try {
                    await fieldsService.createField(collection, diff[0].rhs);
                }
                catch (err) {
                    logger_1.default.error(`Failed to create field "${collection}.${field}"`);
                    throw err;
                }
            }
            if ((diff === null || diff === void 0 ? void 0 : diff[0].kind) === 'E' || (diff === null || diff === void 0 ? void 0 : diff[0].kind) === 'A' || isNestedMetaUpdate(diff === null || diff === void 0 ? void 0 : diff[0])) {
                const newValues = snapshot.fields.find((snapshotField) => {
                    return snapshotField.collection === collection && snapshotField.field === field;
                });
                if (newValues) {
                    try {
                        await fieldsService.updateField(collection, {
                            ...newValues,
                        });
                    }
                    catch (err) {
                        logger_1.default.error(`Failed to update field "${collection}.${field}"`);
                        throw err;
                    }
                }
            }
            if ((diff === null || diff === void 0 ? void 0 : diff[0].kind) === 'D' && !isNestedMetaUpdate(diff === null || diff === void 0 ? void 0 : diff[0])) {
                try {
                    await fieldsService.deleteField(collection, field);
                }
                catch (err) {
                    logger_1.default.error(`Failed to delete field "${collection}.${field}"`);
                    throw err;
                }
                // Field deletion also cleans up the relationship. We should ignore any relationship
                // changes attached to this now non-existing field
                snapshotDiff.relations = snapshotDiff.relations.filter((relation) => (relation.collection === collection && relation.field === field) === false);
            }
        }
        const relationsService = new services_1.RelationsService({ knex: trx, schema: await (0, get_schema_1.getSchema)({ database: trx }) });
        for (const { collection, field, diff } of snapshotDiff.relations) {
            const structure = {};
            for (const diffEdit of diff) {
                (0, lodash_1.set)(structure, diffEdit.path, undefined);
            }
            if ((diff === null || diff === void 0 ? void 0 : diff[0].kind) === 'N') {
                try {
                    await relationsService.createOne(diff[0].rhs);
                }
                catch (err) {
                    logger_1.default.error(`Failed to create relation "${collection}.${field}"`);
                    throw err;
                }
            }
            if ((diff === null || diff === void 0 ? void 0 : diff[0].kind) === 'E' || (diff === null || diff === void 0 ? void 0 : diff[0].kind) === 'A') {
                const newValues = snapshot.relations.find((relation) => {
                    return relation.collection === collection && relation.field === field;
                });
                if (newValues) {
                    try {
                        await relationsService.updateOne(collection, field, newValues);
                    }
                    catch (err) {
                        logger_1.default.error(`Failed to update relation "${collection}.${field}"`);
                        throw err;
                    }
                }
            }
            if ((diff === null || diff === void 0 ? void 0 : diff[0].kind) === 'D') {
                try {
                    await relationsService.deleteOne(collection, field);
                }
                catch (err) {
                    logger_1.default.error(`Failed to delete relation "${collection}.${field}"`);
                    throw err;
                }
            }
        }
    });
    await (systemCache === null || systemCache === void 0 ? void 0 : systemCache.clear());
}
exports.applySnapshot = applySnapshot;
function isNestedMetaUpdate(diff) {
    if (!diff)
        return false;
    if (diff.kind !== 'N' && diff.kind !== 'D')
        return false;
    if (!diff.path || diff.path.length < 2 || diff.path[0] !== 'meta')
        return false;
    return true;
}
exports.isNestedMetaUpdate = isNestedMetaUpdate;
