"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (k !== "default" && Object.prototype.hasOwnProperty.call(mod, k)) __createBinding(result, mod, k);
    __setModuleDefault(result, mod);
    return result;
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.FilesService = void 0;
const format_title_1 = __importDefault(require("@directus/format-title"));
const axios_1 = __importDefault(require("axios"));
const exifr_1 = __importDefault(require("exifr"));
const lodash_1 = require("lodash");
const mime_types_1 = require("mime-types");
const path_1 = __importDefault(require("path"));
const sharp_1 = __importDefault(require("sharp"));
const url_1 = __importStar(require("url"));
const util_1 = require("util");
const dns_1 = require("dns");
const emitter_1 = __importDefault(require("../emitter"));
const env_1 = __importDefault(require("../env"));
const exceptions_1 = require("../exceptions");
const logger_1 = __importDefault(require("../logger"));
const storage_1 = __importDefault(require("../storage"));
const utils_1 = require("@directus/shared/utils");
const items_1 = require("./items");
const net_1 = __importDefault(require("net"));
const os_1 = __importDefault(require("os"));
const encodeurl_1 = __importDefault(require("encodeurl"));
const lookupDNS = (0, util_1.promisify)(dns_1.lookup);
class FilesService extends items_1.ItemsService {
    constructor(options) {
        super('directus_files', options);
    }
    /**
     * Upload a single new file to the configured storage adapter
     */
    async uploadOne(stream, data, primaryKey, opts) {
        var _a, _b, _c, _d, _e, _f;
        const payload = (0, lodash_1.clone)(data);
        if ('folder' in payload === false) {
            const settings = await this.knex.select('storage_default_folder').from('directus_settings').first();
            if (settings === null || settings === void 0 ? void 0 : settings.storage_default_folder) {
                payload.folder = settings.storage_default_folder;
            }
        }
        if (primaryKey !== undefined) {
            await this.updateOne(primaryKey, payload, { emitEvents: false });
            // If the file you're uploading already exists, we'll consider this upload a replace. In that case, we'll
            // delete the previously saved file and thumbnails to ensure they're generated fresh
            const disk = storage_1.default.disk(payload.storage);
            for await (const file of disk.flatList(String(primaryKey))) {
                await disk.delete(file.path);
            }
        }
        else {
            primaryKey = await this.createOne(payload, { emitEvents: false });
        }
        const fileExtension = path_1.default.extname(payload.filename_download) || (payload.type && '.' + (0, mime_types_1.extension)(payload.type)) || '';
        payload.filename_disk = primaryKey + (fileExtension || '');
        if (!payload.type) {
            payload.type = 'application/octet-stream';
        }
        try {
            await storage_1.default.disk(data.storage).put(payload.filename_disk, stream, payload.type);
        }
        catch (err) {
            logger_1.default.warn(`Couldn't save file ${payload.filename_disk}`);
            logger_1.default.warn(err);
            throw new exceptions_1.ServiceUnavailableException(`Couldn't save file ${payload.filename_disk}`, { service: 'files' });
        }
        const { size } = await storage_1.default.disk(data.storage).getStat(payload.filename_disk);
        payload.filesize = size;
        if (['image/jpeg', 'image/png', 'image/webp', 'image/gif', 'image/tiff'].includes(payload.type)) {
            const buffer = await storage_1.default.disk(data.storage).getBuffer(payload.filename_disk);
            const { height, width, description, title, tags, metadata } = await this.getMetadata(buffer.content);
            (_a = payload.height) !== null && _a !== void 0 ? _a : (payload.height = height);
            (_b = payload.width) !== null && _b !== void 0 ? _b : (payload.width = width);
            (_c = payload.description) !== null && _c !== void 0 ? _c : (payload.description = description);
            (_d = payload.title) !== null && _d !== void 0 ? _d : (payload.title = title);
            (_e = payload.tags) !== null && _e !== void 0 ? _e : (payload.tags = tags);
            (_f = payload.metadata) !== null && _f !== void 0 ? _f : (payload.metadata = metadata);
        }
        // We do this in a service without accountability. Even if you don't have update permissions to the file,
        // we still want to be able to set the extracted values from the file on create
        const sudoService = new items_1.ItemsService('directus_files', {
            knex: this.knex,
            schema: this.schema,
        });
        await sudoService.updateOne(primaryKey, payload, { emitEvents: false });
        if (this.cache && env_1.default.CACHE_AUTO_PURGE) {
            await this.cache.clear();
        }
        if ((opts === null || opts === void 0 ? void 0 : opts.emitEvents) !== false) {
            emitter_1.default.emitAction('files.upload', {
                payload,
                key: primaryKey,
                collection: this.collection,
            }, {
                database: this.knex,
                schema: this.schema,
                accountability: this.accountability,
            });
        }
        return primaryKey;
    }
    /**
     * Extract metadata from a buffer's content
     */
    async getMetadata(bufferContent, allowList = env_1.default.FILE_METADATA_ALLOW_LIST) {
        const metadata = {};
        try {
            const sharpMetadata = await (0, sharp_1.default)(bufferContent, {}).metadata();
            if (sharpMetadata.orientation && sharpMetadata.orientation >= 5) {
                metadata.height = sharpMetadata.width;
                metadata.width = sharpMetadata.height;
            }
            else {
                metadata.width = sharpMetadata.width;
                metadata.height = sharpMetadata.height;
            }
        }
        catch (err) {
            logger_1.default.warn(`Couldn't extract sharp metadata from file`);
            logger_1.default.warn(err);
        }
        try {
            const exifrMetadata = await exifr_1.default.parse(bufferContent, {
                icc: false,
                iptc: true,
                ifd1: true,
                interop: true,
                translateValues: true,
                reviveValues: true,
                mergeOutput: false,
            });
            if (allowList === '*' || (allowList === null || allowList === void 0 ? void 0 : allowList[0]) === '*') {
                metadata.metadata = exifrMetadata;
            }
            else {
                metadata.metadata = (0, lodash_1.pick)(exifrMetadata, allowList);
            }
            if (!metadata.description && (exifrMetadata === null || exifrMetadata === void 0 ? void 0 : exifrMetadata.Caption)) {
                metadata.description = exifrMetadata.Caption;
            }
            if (exifrMetadata === null || exifrMetadata === void 0 ? void 0 : exifrMetadata.Headline) {
                metadata.title = exifrMetadata.Headline;
            }
            if (exifrMetadata === null || exifrMetadata === void 0 ? void 0 : exifrMetadata.Keywords) {
                metadata.tags = exifrMetadata.Keywords;
            }
        }
        catch (err) {
            logger_1.default.warn(`Couldn't extract EXIF metadata from file`);
            logger_1.default.warn(err);
        }
        return metadata;
    }
    /**
     * Import a single file from an external URL
     */
    async importOne(importURL, body) {
        var _a, _b, _c;
        const fileCreatePermissions = (_b = (_a = this.accountability) === null || _a === void 0 ? void 0 : _a.permissions) === null || _b === void 0 ? void 0 : _b.find((permission) => permission.collection === 'directus_files' && permission.action === 'create');
        if (this.accountability && ((_c = this.accountability) === null || _c === void 0 ? void 0 : _c.admin) !== true && !fileCreatePermissions) {
            throw new exceptions_1.ForbiddenException();
        }
        let resolvedUrl;
        try {
            resolvedUrl = new url_1.URL(importURL);
        }
        catch (err) {
            logger_1.default.warn(err, `Requested URL ${importURL} isn't a valid URL`);
            throw new exceptions_1.ServiceUnavailableException(`Couldn't fetch file from url "${importURL}"`, {
                service: 'external-file',
            });
        }
        let ip = resolvedUrl.hostname;
        if (net_1.default.isIP(ip) === 0) {
            try {
                ip = (await lookupDNS(ip)).address;
            }
            catch (err) {
                logger_1.default.warn(err, `Couldn't lookup the DNS for url ${importURL}`);
                throw new exceptions_1.ServiceUnavailableException(`Couldn't fetch file from url "${importURL}"`, {
                    service: 'external-file',
                });
            }
        }
        if (env_1.default.IMPORT_IP_DENY_LIST.includes('0.0.0.0')) {
            const networkInterfaces = os_1.default.networkInterfaces();
            for (const networkInfo of Object.values(networkInterfaces)) {
                if (!networkInfo)
                    continue;
                for (const info of networkInfo) {
                    if (info.address === ip) {
                        logger_1.default.warn(`Requested URL ${importURL} resolves to localhost.`);
                        throw new exceptions_1.ServiceUnavailableException(`Couldn't fetch file from url "${importURL}"`, {
                            service: 'external-file',
                        });
                    }
                }
            }
        }
        if (env_1.default.IMPORT_IP_DENY_LIST.includes(ip)) {
            logger_1.default.warn(`Requested URL ${importURL} resolves to a denied IP address.`);
            throw new exceptions_1.ServiceUnavailableException(`Couldn't fetch file from url "${importURL}"`, {
                service: 'external-file',
            });
        }
        let fileResponse;
        try {
            fileResponse = await axios_1.default.get((0, encodeurl_1.default)(importURL), {
                responseType: 'stream',
            });
        }
        catch (err) {
            logger_1.default.warn(err, `Couldn't fetch file from url "${importURL}"`);
            throw new exceptions_1.ServiceUnavailableException(`Couldn't fetch file from url "${importURL}"`, {
                service: 'external-file',
            });
        }
        const parsedURL = url_1.default.parse(fileResponse.request.res.responseUrl);
        const filename = decodeURI(path_1.default.basename(parsedURL.pathname));
        const payload = {
            filename_download: filename,
            storage: (0, utils_1.toArray)(env_1.default.STORAGE_LOCATIONS)[0],
            type: fileResponse.headers['content-type'],
            title: (0, format_title_1.default)(filename),
            ...(body || {}),
        };
        return await this.uploadOne(fileResponse.data, payload);
    }
    /**
     * Create a file (only applicable when it is not a multipart/data POST request)
     * Useful for associating metadata with existing file in storage
     */
    async createOne(data, opts) {
        if (!data.type) {
            throw new exceptions_1.InvalidPayloadException(`"type" is required`);
        }
        const key = await super.createOne(data, opts);
        return key;
    }
    /**
     * Delete a file
     */
    async deleteOne(key, opts) {
        await this.deleteMany([key], opts);
        return key;
    }
    /**
     * Delete multiple files
     */
    async deleteMany(keys, opts) {
        const files = await super.readMany(keys, { fields: ['id', 'storage'], limit: -1 });
        if (!files) {
            throw new exceptions_1.ForbiddenException();
        }
        await super.deleteMany(keys);
        for (const file of files) {
            const disk = storage_1.default.disk(file.storage);
            // Delete file + thumbnails
            for await (const { path } of disk.flatList(file.id)) {
                await disk.delete(path);
            }
        }
        if (this.cache && env_1.default.CACHE_AUTO_PURGE && (opts === null || opts === void 0 ? void 0 : opts.autoPurgeCache) !== false) {
            await this.cache.clear();
        }
        return keys;
    }
}
exports.FilesService = FilesService;
