"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const path_1 = __importDefault(require("path"));
const chalk_1 = __importDefault(require("chalk"));
const fs_extra_1 = __importDefault(require("fs-extra"));
const execa_1 = __importDefault(require("execa"));
const ora_1 = __importDefault(require("ora"));
const constants_1 = require("@directus/shared/constants");
const utils_1 = require("@directus/shared/utils");
const logger_1 = require("../utils/logger");
const languages_1 = require("../utils/languages");
const get_sdk_version_1 = __importDefault(require("../utils/get-sdk-version"));
const get_extension_dev_deps_1 = __importDefault(require("./helpers/get-extension-dev-deps"));
const copy_template_1 = __importDefault(require("./helpers/copy-template"));
const get_package_manager_1 = __importDefault(require("../utils/get-package-manager"));
async function create(type, name, options) {
    var _a;
    const targetDir = name.substring(name.lastIndexOf('/') + 1);
    const targetPath = path_1.default.resolve(targetDir);
    if (!(0, utils_1.isIn)(type, constants_1.EXTENSION_PACKAGE_TYPES)) {
        (0, logger_1.log)(`Extension type ${chalk_1.default.bold(type)} is not supported. Available extension types: ${constants_1.EXTENSION_PACKAGE_TYPES.map((t) => chalk_1.default.bold.magenta(t)).join(', ')}.`, 'error');
        process.exit(1);
    }
    if (targetDir.length === 0) {
        (0, logger_1.log)(`Extension name can not be empty.`, 'error');
        process.exit(1);
    }
    if (await fs_extra_1.default.pathExists(targetPath)) {
        const info = await fs_extra_1.default.stat(targetPath);
        if (!info.isDirectory()) {
            (0, logger_1.log)(`Destination ${chalk_1.default.bold(targetDir)} already exists and is not a directory.`, 'error');
            process.exit(1);
        }
        const files = await fs_extra_1.default.readdir(targetPath);
        if (files.length > 0) {
            (0, logger_1.log)(`Destination ${chalk_1.default.bold(targetDir)} already exists and is not empty.`, 'error');
            process.exit(1);
        }
    }
    if ((0, utils_1.isIn)(type, constants_1.PACKAGE_EXTENSION_TYPES)) {
        await createPackageExtension({ type, name, targetDir, targetPath });
    }
    else {
        const language = (_a = options.language) !== null && _a !== void 0 ? _a : 'javascript';
        await createLocalExtension({ type, name, targetDir, targetPath, language });
    }
}
exports.default = create;
async function createPackageExtension({ type, name, targetDir, targetPath, }) {
    const spinner = (0, ora_1.default)(chalk_1.default.bold('Scaffolding Directus extension...')).start();
    await fs_extra_1.default.ensureDir(targetPath);
    await (0, copy_template_1.default)(type, targetPath);
    const host = `^${(0, get_sdk_version_1.default)()}`;
    const options = type === 'bundle' ? { type, path: { app: 'dist/app.js', api: 'dist/api.js' }, entries: [], host } : { type, host };
    const packageManifest = getPackageManifest(name, options, await (0, get_extension_dev_deps_1.default)(type));
    await fs_extra_1.default.writeJSON(path_1.default.join(targetPath, 'package.json'), packageManifest, { spaces: '\t' });
    const packageManager = (0, get_package_manager_1.default)();
    await (0, execa_1.default)(packageManager, ['install'], { cwd: targetPath });
    spinner.succeed(chalk_1.default.bold('Done'));
    (0, logger_1.log)(getDoneMessage(type, targetDir, targetPath, packageManager));
}
async function createLocalExtension({ type, name, targetDir, targetPath, language, }) {
    if (!(0, languages_1.isLanguage)(language)) {
        (0, logger_1.log)(`Language ${chalk_1.default.bold(language)} is not supported. Available languages: ${constants_1.EXTENSION_LANGUAGES.map((t) => chalk_1.default.bold.magenta(t)).join(', ')}.`, 'error');
        process.exit(1);
    }
    const spinner = (0, ora_1.default)(chalk_1.default.bold('Scaffolding Directus extension...')).start();
    await fs_extra_1.default.ensureDir(targetPath);
    await (0, copy_template_1.default)(type, targetPath, 'src', language);
    const host = `^${(0, get_sdk_version_1.default)()}`;
    const options = (0, utils_1.isIn)(type, constants_1.HYBRID_EXTENSION_TYPES)
        ? {
            type,
            path: { app: 'dist/app.js', api: 'dist/api.js' },
            source: { app: `src/app.${(0, languages_1.languageToShort)(language)}`, api: `src/api.${(0, languages_1.languageToShort)(language)}` },
            host,
        }
        : {
            type,
            path: 'dist/index.js',
            source: `src/index.${(0, languages_1.languageToShort)(language)}`,
            host,
        };
    const packageManifest = getPackageManifest(name, options, await (0, get_extension_dev_deps_1.default)(type, language));
    await fs_extra_1.default.writeJSON(path_1.default.join(targetPath, 'package.json'), packageManifest, { spaces: '\t' });
    const packageManager = (0, get_package_manager_1.default)();
    await (0, execa_1.default)(packageManager, ['install'], { cwd: targetPath });
    spinner.succeed(chalk_1.default.bold('Done'));
    (0, logger_1.log)(getDoneMessage(type, targetDir, targetPath, packageManager));
}
function getPackageManifest(name, options, deps) {
    return {
        name: constants_1.EXTENSION_NAME_REGEX.test(name) ? name : `directus-extension-${name}`,
        version: '1.0.0',
        keywords: ['directus', 'directus-extension', `directus-custom-${options.type}`],
        [constants_1.EXTENSION_PKG_KEY]: options,
        scripts: {
            build: 'directus-extension build',
            dev: 'directus-extension build -w --no-minify',
        },
        devDependencies: deps,
    };
}
function getDoneMessage(type, targetDir, targetPath, packageManager) {
    return `
Your ${type} extension has been created at ${chalk_1.default.green(targetPath)}

To start developing, run:
	${chalk_1.default.blue('cd')} ${targetDir}
	${chalk_1.default.blue(`${packageManager} run`)} dev

and then to build for production, run:
	${chalk_1.default.blue(`${packageManager} run`)} build
`;
}
