"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const constants_1 = require("@directus/shared/constants");
const utils_1 = require("@directus/shared/utils");
const plugin_commonjs_1 = __importDefault(require("@rollup/plugin-commonjs"));
const plugin_json_1 = __importDefault(require("@rollup/plugin-json"));
const plugin_node_resolve_1 = require("@rollup/plugin-node-resolve");
const plugin_replace_1 = __importDefault(require("@rollup/plugin-replace"));
const plugin_terser_1 = __importDefault(require("@rollup/plugin-terser"));
const plugin_virtual_1 = __importDefault(require("@rollup/plugin-virtual"));
const chalk_1 = __importDefault(require("chalk"));
const fs_extra_1 = __importDefault(require("fs-extra"));
const ora_1 = __importDefault(require("ora"));
const path_1 = __importDefault(require("path"));
const rollup_1 = require("rollup");
const rollup_plugin_styles_1 = __importDefault(require("rollup-plugin-styles"));
const rollup_plugin_typescript2_1 = __importDefault(require("rollup-plugin-typescript2"));
const rollup_plugin_vue_1 = __importDefault(require("rollup-plugin-vue"));
const languages_1 = require("../utils/languages");
const logger_1 = require("../utils/logger");
const try_parse_json_1 = __importDefault(require("../utils/try-parse-json"));
const generate_bundle_entrypoint_1 = __importDefault(require("./helpers/generate-bundle-entrypoint"));
const load_config_1 = __importDefault(require("./helpers/load-config"));
const validate_cli_options_1 = require("./helpers/validate-cli-options");
async function build(options) {
    var _a, _b, _c;
    const watch = (_a = options.watch) !== null && _a !== void 0 ? _a : false;
    const sourcemap = (_b = options.sourcemap) !== null && _b !== void 0 ? _b : false;
    const minify = (_c = options.minify) !== null && _c !== void 0 ? _c : false;
    if (!options.type && !options.input && !options.output) {
        const packagePath = path_1.default.resolve('package.json');
        if (!(await fs_extra_1.default.pathExists(packagePath))) {
            (0, logger_1.log)(`Current directory is not a valid package.`, 'error');
            process.exit(1);
        }
        const extensionManifest = await fs_extra_1.default.readJSON(packagePath);
        if (!(0, utils_1.validateExtensionManifest)(extensionManifest)) {
            (0, logger_1.log)(`Current directory is not a valid Directus extension.`, 'error');
            process.exit(1);
        }
        const extensionOptions = extensionManifest[constants_1.EXTENSION_PKG_KEY];
        if (extensionOptions.type === 'pack') {
            (0, logger_1.log)(`Building extension type ${chalk_1.default.bold('pack')} is not currently supported.`, 'error');
            process.exit(1);
        }
        if (extensionOptions.type === 'bundle') {
            await buildBundleExtension({
                entries: extensionOptions.entries,
                outputApp: extensionOptions.path.app,
                outputApi: extensionOptions.path.api,
                watch,
                sourcemap,
                minify,
            });
        }
        else if ((0, utils_1.isTypeIn)(extensionOptions, constants_1.HYBRID_EXTENSION_TYPES)) {
            await buildHybridExtension({
                inputApp: extensionOptions.source.app,
                inputApi: extensionOptions.source.api,
                outputApp: extensionOptions.path.app,
                outputApi: extensionOptions.path.api,
                watch,
                sourcemap,
                minify,
            });
        }
        else {
            await buildAppOrApiExtension({
                type: extensionOptions.type,
                input: extensionOptions.source,
                output: extensionOptions.path,
                watch,
                sourcemap,
                minify,
            });
        }
    }
    else {
        const type = options.type;
        const input = options.input;
        const output = options.output;
        if (!type) {
            (0, logger_1.log)(`Extension type has to be specified using the ${chalk_1.default.blue('[-t, --type <type>]')} option.`, 'error');
            process.exit(1);
        }
        if (!(0, utils_1.isIn)(type, constants_1.EXTENSION_PACKAGE_TYPES)) {
            (0, logger_1.log)(`Extension type ${chalk_1.default.bold(type)} is not supported. Available extension types: ${constants_1.EXTENSION_PACKAGE_TYPES.map((t) => chalk_1.default.bold.magenta(t)).join(', ')}.`, 'error');
            process.exit(1);
        }
        if (type === 'pack') {
            (0, logger_1.log)(`Building extension type ${chalk_1.default.bold('pack')} is not currently supported.`, 'error');
            process.exit(1);
        }
        if (!input) {
            (0, logger_1.log)(`Extension entrypoint has to be specified using the ${chalk_1.default.blue('[-i, --input <file>]')} option.`, 'error');
            process.exit(1);
        }
        if (!output) {
            (0, logger_1.log)(`Extension output file has to be specified using the ${chalk_1.default.blue('[-o, --output <file>]')} option.`, 'error');
            process.exit(1);
        }
        if (type === 'bundle') {
            const entries = (0, try_parse_json_1.default)(input);
            const splitOutput = (0, try_parse_json_1.default)(output);
            if (!(0, validate_cli_options_1.validateBundleEntriesOption)(entries)) {
                (0, logger_1.log)(`Input option needs to be of the format ${chalk_1.default.blue(`[-i '[{"type":"<extension-type>","name":"<extension-name>","source":<entrypoint>}]']`)}.`, 'error');
                process.exit(1);
            }
            if (!(0, validate_cli_options_1.validateSplitEntrypointOption)(splitOutput)) {
                (0, logger_1.log)(`Output option needs to be of the format ${chalk_1.default.blue(`[-o '{"app":"<app-entrypoint>","api":"<api-entrypoint>"}']`)}.`, 'error');
                process.exit(1);
            }
            await buildBundleExtension({
                entries,
                outputApp: splitOutput.app,
                outputApi: splitOutput.api,
                watch,
                sourcemap,
                minify,
            });
        }
        else if ((0, utils_1.isIn)(type, constants_1.HYBRID_EXTENSION_TYPES)) {
            const splitInput = (0, try_parse_json_1.default)(input);
            const splitOutput = (0, try_parse_json_1.default)(output);
            if (!(0, validate_cli_options_1.validateSplitEntrypointOption)(splitInput)) {
                (0, logger_1.log)(`Input option needs to be of the format ${chalk_1.default.blue(`[-i '{"app":"<app-entrypoint>","api":"<api-entrypoint>"}']`)}.`, 'error');
                process.exit(1);
            }
            if (!(0, validate_cli_options_1.validateSplitEntrypointOption)(splitOutput)) {
                (0, logger_1.log)(`Output option needs to be of the format ${chalk_1.default.blue(`[-o '{"app":"<app-entrypoint>","api":"<api-entrypoint>"}']`)}.`, 'error');
                process.exit(1);
            }
            await buildHybridExtension({
                inputApp: splitInput.app,
                inputApi: splitInput.api,
                outputApp: splitOutput.app,
                outputApi: splitOutput.api,
                watch,
                sourcemap,
                minify,
            });
        }
        else {
            await buildAppOrApiExtension({
                type,
                input,
                output,
                watch,
                sourcemap,
                minify,
            });
        }
    }
}
exports.default = build;
async function buildAppOrApiExtension({ type, input, output, watch, sourcemap, minify, }) {
    var _a;
    if (!(await fs_extra_1.default.pathExists(input)) || !(await fs_extra_1.default.stat(input)).isFile()) {
        (0, logger_1.log)(`Entrypoint ${chalk_1.default.bold(input)} does not exist.`, 'error');
        process.exit(1);
    }
    if (output.length === 0) {
        (0, logger_1.log)(`Output file can not be empty.`, 'error');
        process.exit(1);
    }
    const language = (0, languages_1.getLanguageFromPath)(input);
    if (!(0, languages_1.isLanguage)(language)) {
        (0, logger_1.log)(`Language ${chalk_1.default.bold(language)} is not supported.`, 'error');
        process.exit(1);
    }
    const config = await (0, load_config_1.default)();
    const plugins = (_a = config.plugins) !== null && _a !== void 0 ? _a : [];
    const mode = (0, utils_1.isIn)(type, constants_1.APP_EXTENSION_TYPES) ? 'browser' : 'node';
    const rollupOptions = getRollupOptions({ mode, input, language, sourcemap, minify, plugins });
    const rollupOutputOptions = getRollupOutputOptions({ mode, output, sourcemap });
    if (watch) {
        await watchExtension({ rollupOptions, rollupOutputOptions });
    }
    else {
        await buildExtension({ rollupOptions, rollupOutputOptions });
    }
}
async function buildHybridExtension({ inputApp, inputApi, outputApp, outputApi, watch, sourcemap, minify, }) {
    var _a;
    if (!(await fs_extra_1.default.pathExists(inputApp)) || !(await fs_extra_1.default.stat(inputApp)).isFile()) {
        (0, logger_1.log)(`App entrypoint ${chalk_1.default.bold(inputApp)} does not exist.`, 'error');
        process.exit(1);
    }
    if (!(await fs_extra_1.default.pathExists(inputApi)) || !(await fs_extra_1.default.stat(inputApi)).isFile()) {
        (0, logger_1.log)(`API entrypoint ${chalk_1.default.bold(inputApi)} does not exist.`, 'error');
        process.exit(1);
    }
    if (outputApp.length === 0) {
        (0, logger_1.log)(`App output file can not be empty.`, 'error');
        process.exit(1);
    }
    if (outputApi.length === 0) {
        (0, logger_1.log)(`API output file can not be empty.`, 'error');
        process.exit(1);
    }
    const languageApp = (0, languages_1.getLanguageFromPath)(inputApp);
    const languageApi = (0, languages_1.getLanguageFromPath)(inputApi);
    if (!(0, languages_1.isLanguage)(languageApp)) {
        (0, logger_1.log)(`App language ${chalk_1.default.bold(languageApp)} is not supported.`, 'error');
        process.exit(1);
    }
    if (!(0, languages_1.isLanguage)(languageApi)) {
        (0, logger_1.log)(`API language ${chalk_1.default.bold(languageApi)} is not supported.`, 'error');
        process.exit(1);
    }
    const config = await (0, load_config_1.default)();
    const plugins = (_a = config.plugins) !== null && _a !== void 0 ? _a : [];
    const rollupOptionsApp = getRollupOptions({
        mode: 'browser',
        input: inputApp,
        language: languageApp,
        sourcemap,
        minify,
        plugins,
    });
    const rollupOptionsApi = getRollupOptions({
        mode: 'node',
        input: inputApi,
        language: languageApi,
        sourcemap,
        minify,
        plugins,
    });
    const rollupOutputOptionsApp = getRollupOutputOptions({ mode: 'browser', output: outputApp, sourcemap });
    const rollupOutputOptionsApi = getRollupOutputOptions({ mode: 'node', output: outputApi, sourcemap });
    const rollupOptionsAll = [
        { rollupOptions: rollupOptionsApp, rollupOutputOptions: rollupOutputOptionsApp },
        { rollupOptions: rollupOptionsApi, rollupOutputOptions: rollupOutputOptionsApi },
    ];
    if (watch) {
        await watchExtension(rollupOptionsAll);
    }
    else {
        await buildExtension(rollupOptionsAll);
    }
}
async function buildBundleExtension({ entries, outputApp, outputApi, watch, sourcemap, minify, }) {
    var _a;
    if (outputApp.length === 0) {
        (0, logger_1.log)(`App output file can not be empty.`, 'error');
        process.exit(1);
    }
    if (outputApi.length === 0) {
        (0, logger_1.log)(`API output file can not be empty.`, 'error');
        process.exit(1);
    }
    const languagesApp = new Set();
    const languagesApi = new Set();
    for (const entry of entries) {
        if ((0, utils_1.isTypeIn)(entry, constants_1.HYBRID_EXTENSION_TYPES)) {
            const inputApp = entry.source.app;
            const inputApi = entry.source.api;
            if (!(await fs_extra_1.default.pathExists(inputApp)) || !(await fs_extra_1.default.stat(inputApp)).isFile()) {
                (0, logger_1.log)(`App entrypoint ${chalk_1.default.bold(inputApp)} does not exist.`, 'error');
                process.exit(1);
            }
            if (!(await fs_extra_1.default.pathExists(inputApi)) || !(await fs_extra_1.default.stat(inputApi)).isFile()) {
                (0, logger_1.log)(`API entrypoint ${chalk_1.default.bold(inputApi)} does not exist.`, 'error');
                process.exit(1);
            }
            const languageApp = (0, languages_1.getLanguageFromPath)(inputApp);
            const languageApi = (0, languages_1.getLanguageFromPath)(inputApi);
            if (!(0, languages_1.isLanguage)(languageApp)) {
                (0, logger_1.log)(`App language ${chalk_1.default.bold(languageApp)} is not supported.`, 'error');
                process.exit(1);
            }
            if (!(0, languages_1.isLanguage)(languageApi)) {
                (0, logger_1.log)(`API language ${chalk_1.default.bold(languageApi)} is not supported.`, 'error');
                process.exit(1);
            }
            languagesApp.add(languageApp);
            languagesApi.add(languageApi);
        }
        else {
            const input = entry.source;
            if (!(await fs_extra_1.default.pathExists(input)) || !(await fs_extra_1.default.stat(input)).isFile()) {
                (0, logger_1.log)(`Entrypoint ${chalk_1.default.bold(input)} does not exist.`, 'error');
                process.exit(1);
            }
            const language = (0, languages_1.getLanguageFromPath)(input);
            if (!(0, languages_1.isLanguage)(language)) {
                (0, logger_1.log)(`Language ${chalk_1.default.bold(language)} is not supported.`, 'error');
                process.exit(1);
            }
            if ((0, utils_1.isIn)(entry.type, constants_1.APP_EXTENSION_TYPES)) {
                languagesApp.add(language);
            }
            else {
                languagesApi.add(language);
            }
        }
    }
    const config = await (0, load_config_1.default)();
    const plugins = (_a = config.plugins) !== null && _a !== void 0 ? _a : [];
    const entrypointApp = (0, generate_bundle_entrypoint_1.default)('app', entries);
    const entrypointApi = (0, generate_bundle_entrypoint_1.default)('api', entries);
    const rollupOptionsApp = getRollupOptions({
        mode: 'browser',
        input: { entry: entrypointApp },
        language: Array.from(languagesApp),
        sourcemap,
        minify,
        plugins,
    });
    const rollupOptionsApi = getRollupOptions({
        mode: 'node',
        input: { entry: entrypointApi },
        language: Array.from(languagesApi),
        sourcemap,
        minify,
        plugins,
    });
    const rollupOutputOptionsApp = getRollupOutputOptions({ mode: 'browser', output: outputApp, sourcemap });
    const rollupOutputOptionsApi = getRollupOutputOptions({ mode: 'node', output: outputApi, sourcemap });
    const rollupOptionsAll = [
        { rollupOptions: rollupOptionsApp, rollupOutputOptions: rollupOutputOptionsApp },
        { rollupOptions: rollupOptionsApi, rollupOutputOptions: rollupOutputOptionsApi },
    ];
    if (watch) {
        await watchExtension(rollupOptionsAll);
    }
    else {
        await buildExtension(rollupOptionsAll);
    }
}
async function buildExtension(config) {
    const configs = Array.isArray(config) ? config : [config];
    const spinner = (0, ora_1.default)(chalk_1.default.bold('Building Directus extension...')).start();
    const result = await Promise.all(configs.map(async (c) => {
        try {
            const bundle = await (0, rollup_1.rollup)(c.rollupOptions);
            await bundle.write(c.rollupOutputOptions);
            await bundle.close();
        }
        catch (error) {
            return formatRollupError(error);
        }
        return null;
    }));
    const resultErrors = result.filter((r) => r !== null);
    if (resultErrors.length > 0) {
        spinner.fail(chalk_1.default.bold('Failed'));
        (0, logger_1.log)(resultErrors.join('\n\n'));
        process.exit(1);
    }
    else {
        spinner.succeed(chalk_1.default.bold('Done'));
    }
}
async function watchExtension(config) {
    const configs = Array.isArray(config) ? config : [config];
    const spinner = (0, ora_1.default)(chalk_1.default.bold('Building Directus extension...'));
    let buildCount = 0;
    for (const c of configs) {
        const watcher = (0, rollup_1.watch)({
            ...c.rollupOptions,
            output: c.rollupOutputOptions,
        });
        watcher.on('event', async (event) => {
            switch (event.code) {
                case 'BUNDLE_START':
                    if (buildCount === 0) {
                        (0, logger_1.clear)();
                        spinner.start();
                    }
                    buildCount++;
                    break;
                case 'BUNDLE_END':
                    await event.result.close();
                    buildCount--;
                    if (buildCount === 0) {
                        spinner.succeed(chalk_1.default.bold('Done'));
                        (0, logger_1.log)(chalk_1.default.bold.green('Watching files for changes...'));
                    }
                    break;
                case 'ERROR': {
                    buildCount--;
                    spinner.fail(chalk_1.default.bold('Failed'));
                    (0, logger_1.log)(formatRollupError(event.error));
                    if (buildCount > 0) {
                        spinner.start();
                    }
                    break;
                }
            }
        });
    }
}
function getRollupOptions({ mode, input, language, sourcemap, minify, plugins, }) {
    const languages = Array.isArray(language) ? language : [language];
    if (mode === 'browser') {
        return {
            input: typeof input !== 'string' ? 'entry' : input,
            external: constants_1.APP_SHARED_DEPS,
            plugins: [
                typeof input !== 'string' ? (0, plugin_virtual_1.default)(input) : null,
                (0, rollup_plugin_vue_1.default)({ preprocessStyles: true }),
                languages.includes('typescript') ? (0, rollup_plugin_typescript2_1.default)({ check: false }) : null,
                (0, rollup_plugin_styles_1.default)(),
                ...plugins,
                (0, plugin_node_resolve_1.nodeResolve)({ browser: true }),
                (0, plugin_commonjs_1.default)({ esmExternals: true, sourceMap: sourcemap }),
                (0, plugin_json_1.default)(),
                (0, plugin_replace_1.default)({
                    values: {
                        'process.env.NODE_ENV': JSON.stringify('production'),
                    },
                    preventAssignment: true,
                }),
                minify ? (0, plugin_terser_1.default)() : null,
            ],
        };
    }
    else {
        return {
            input: typeof input !== 'string' ? 'entry' : input,
            external: constants_1.API_SHARED_DEPS,
            plugins: [
                typeof input !== 'string' ? (0, plugin_virtual_1.default)(input) : null,
                languages.includes('typescript') ? (0, rollup_plugin_typescript2_1.default)({ check: false }) : null,
                ...plugins,
                (0, plugin_node_resolve_1.nodeResolve)(),
                (0, plugin_commonjs_1.default)({ sourceMap: sourcemap }),
                (0, plugin_json_1.default)(),
                (0, plugin_replace_1.default)({
                    values: {
                        'process.env.NODE_ENV': JSON.stringify('production'),
                    },
                    preventAssignment: true,
                }),
                minify ? (0, plugin_terser_1.default)() : null,
            ],
        };
    }
}
function getRollupOutputOptions({ mode, output, sourcemap, }) {
    if (mode === 'browser') {
        return {
            file: output,
            format: 'es',
            inlineDynamicImports: true,
            sourcemap,
        };
    }
    else {
        return {
            file: output,
            format: 'cjs',
            exports: 'auto',
            inlineDynamicImports: true,
            sourcemap,
        };
    }
}
function formatRollupError(error) {
    var _a;
    let message = '';
    message += `${chalk_1.default.bold.red(`[${error.name}]`)} ${error.message}${error.plugin ? ` (plugin ${error.plugin})` : ''}\n`;
    if (error.url) {
        message += '\n' + chalk_1.default.green(error.url);
    }
    if (error.loc) {
        message += '\n' + chalk_1.default.green(`${(_a = error.loc.file) !== null && _a !== void 0 ? _a : error.id}:${error.loc.line}:${error.loc.column}`);
    }
    else if (error.id) {
        message += '\n' + chalk_1.default.green(error.id);
    }
    if (error.frame) {
        message += '\n' + chalk_1.default.dim(error.frame);
    }
    if (error.stack) {
        message += '\n' + chalk_1.default.dim(error.stack);
    }
    return message;
}
