"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const path_1 = __importDefault(require("path"));
const chalk_1 = __importDefault(require("chalk"));
const fs_extra_1 = __importDefault(require("fs-extra"));
const inquirer_1 = __importDefault(require("inquirer"));
const logger_1 = require("../utils/logger");
const utils_1 = require("@directus/shared/utils");
const node_1 = require("@directus/shared/utils/node");
const constants_1 = require("@directus/shared/constants");
const languages_1 = require("../utils/languages");
const get_extension_dev_deps_1 = __importDefault(require("./helpers/get-extension-dev-deps"));
const execa_1 = __importDefault(require("execa"));
const ora_1 = __importDefault(require("ora"));
const copy_template_1 = __importDefault(require("./helpers/copy-template"));
const detect_json_indent_1 = __importDefault(require("../utils/detect-json-indent"));
const get_package_manager_1 = __importDefault(require("../utils/get-package-manager"));
async function add() {
    var _a, _b;
    const extensionPath = process.cwd();
    const packagePath = path_1.default.resolve('package.json');
    if (!(await fs_extra_1.default.pathExists(packagePath))) {
        (0, logger_1.log)(`Current directory is not a valid package.`, 'error');
        process.exit(1);
    }
    const extensionManifestFile = await fs_extra_1.default.readFile(packagePath, 'utf8');
    const extensionManifest = JSON.parse(extensionManifestFile);
    const indent = (0, detect_json_indent_1.default)(extensionManifestFile);
    if (!(0, utils_1.validateExtensionManifest)(extensionManifest)) {
        (0, logger_1.log)(`Current directory is not a valid Directus extension.`, 'error');
        process.exit(1);
    }
    const extensionOptions = extensionManifest[constants_1.EXTENSION_PKG_KEY];
    if (extensionOptions.type === 'pack') {
        (0, logger_1.log)(`Adding entries to extensions with type ${chalk_1.default.bold('pack')} is not currently supported.`, 'error');
        process.exit(1);
    }
    const sourceExists = await fs_extra_1.default.pathExists(path_1.default.resolve('src'));
    if (extensionOptions.type === 'bundle') {
        const { type, name, language, alternativeSource } = await inquirer_1.default.prompt([
            {
                type: 'list',
                name: 'type',
                message: 'Choose the extension type',
                choices: constants_1.EXTENSION_TYPES,
            },
            {
                type: 'input',
                name: 'name',
                message: 'Choose a name for the entry',
                validate: (name) => (name.length === 0 ? 'Entry name can not be empty.' : true),
            },
            {
                type: 'list',
                name: 'language',
                message: 'Choose the language to use',
                choices: constants_1.EXTENSION_LANGUAGES,
            },
            {
                type: 'input',
                name: 'alternativeSource',
                message: 'Specify the path to the extension source',
                when: !sourceExists && extensionOptions.entries.length > 0,
            },
        ]);
        const spinner = (0, ora_1.default)(chalk_1.default.bold('Modifying Directus extension...')).start();
        const source = alternativeSource !== null && alternativeSource !== void 0 ? alternativeSource : 'src';
        const sourcePath = path_1.default.resolve(source, name);
        await fs_extra_1.default.ensureDir(sourcePath);
        await (0, copy_template_1.default)(type, extensionPath, sourcePath, language);
        const newEntries = [
            ...extensionOptions.entries,
            (0, utils_1.isIn)(type, constants_1.HYBRID_EXTENSION_TYPES)
                ? {
                    type,
                    name,
                    source: {
                        app: `${(0, node_1.pathToRelativeUrl)(source)}/${name}/app.${(0, languages_1.languageToShort)(language)}`,
                        api: `${(0, node_1.pathToRelativeUrl)(source)}/${name}/api.${(0, languages_1.languageToShort)(language)}`,
                    },
                }
                : {
                    type,
                    name,
                    source: `${(0, node_1.pathToRelativeUrl)(source)}/${name}/index.${(0, languages_1.languageToShort)(language)}`,
                },
        ];
        const newExtensionOptions = { ...extensionOptions, entries: newEntries };
        const newExtensionManifest = {
            ...extensionManifest,
            [constants_1.EXTENSION_PKG_KEY]: newExtensionOptions,
            devDependencies: await (0, get_extension_dev_deps_1.default)(newEntries.map((entry) => entry.type), getLanguageFromEntries(newEntries)),
        };
        await fs_extra_1.default.writeJSON(packagePath, newExtensionManifest, { spaces: indent !== null && indent !== void 0 ? indent : '\t' });
        const packageManager = (0, get_package_manager_1.default)();
        await (0, execa_1.default)(packageManager, ['install'], { cwd: extensionPath });
        spinner.succeed(chalk_1.default.bold('Done'));
    }
    else {
        const { proceed } = await inquirer_1.default.prompt([
            {
                type: 'confirm',
                name: 'proceed',
                message: 'This will convert your extension to a bundle. Do you want to proceed?',
            },
        ]);
        if (!proceed) {
            (0, logger_1.log)(`Extension has not been modified.`, 'info');
            process.exit(1);
        }
        const oldName = (_b = (_a = extensionManifest.name.match(constants_1.EXTENSION_NAME_REGEX)) === null || _a === void 0 ? void 0 : _a[1]) !== null && _b !== void 0 ? _b : extensionManifest.name;
        const { type, name, language, convertName, extensionName, alternativeSource } = await inquirer_1.default.prompt([
            {
                type: 'list',
                name: 'type',
                message: 'Choose the extension type',
                choices: constants_1.EXTENSION_TYPES,
            },
            {
                type: 'input',
                name: 'name',
                message: 'Choose a name for the entry',
                validate: (name) => (name.length === 0 ? 'Entry name can not be empty.' : true),
            },
            {
                type: 'list',
                name: 'language',
                message: 'Choose the language to use',
                choices: constants_1.EXTENSION_LANGUAGES,
            },
            {
                type: 'input',
                name: 'convertName',
                message: 'Choose a name for the extension that is converted to an entry',
                default: oldName,
                validate: (name) => (name.length === 0 ? 'Entry name can not be empty.' : true),
            },
            {
                type: 'input',
                name: 'extensionName',
                message: 'Choose a name for the extension',
                default: ({ convertName }) => (convertName !== oldName ? oldName : null),
                validate: (name) => (name.length === 0 ? 'Extension name can not be empty.' : true),
            },
            {
                type: 'input',
                name: 'alternativeSource',
                message: 'Specify the path to the extension source',
                when: !sourceExists,
            },
        ]);
        const spinner = (0, ora_1.default)(chalk_1.default.bold('Modifying Directus extension...')).start();
        const source = alternativeSource !== null && alternativeSource !== void 0 ? alternativeSource : 'src';
        const convertSourcePath = path_1.default.resolve(source, convertName);
        const entrySourcePath = path_1.default.resolve(source, name);
        const convertFiles = await fs_extra_1.default.readdir(source);
        await Promise.all(convertFiles.map((file) => fs_extra_1.default.move(path_1.default.resolve(source, file), path_1.default.join(convertSourcePath, file))));
        await fs_extra_1.default.ensureDir(entrySourcePath);
        await (0, copy_template_1.default)(type, extensionPath, entrySourcePath, language);
        const toConvertSourceUrl = (entrypoint) => path_1.default.posix.join((0, node_1.pathToRelativeUrl)(source), convertName, path_1.default.posix.relative(source, entrypoint));
        const entries = [
            (0, utils_1.isTypeIn)(extensionOptions, constants_1.HYBRID_EXTENSION_TYPES)
                ? {
                    type: extensionOptions.type,
                    name: convertName,
                    source: {
                        app: toConvertSourceUrl(extensionOptions.source.app),
                        api: toConvertSourceUrl(extensionOptions.source.api),
                    },
                }
                : {
                    type: extensionOptions.type,
                    name: convertName,
                    source: toConvertSourceUrl(extensionOptions.source),
                },
            (0, utils_1.isIn)(type, constants_1.HYBRID_EXTENSION_TYPES)
                ? {
                    type,
                    name,
                    source: {
                        app: `${(0, node_1.pathToRelativeUrl)(source)}/${name}/app.${(0, languages_1.languageToShort)(language)}`,
                        api: `${(0, node_1.pathToRelativeUrl)(source)}/${name}/api.${(0, languages_1.languageToShort)(language)}`,
                    },
                }
                : {
                    type,
                    name,
                    source: `${(0, node_1.pathToRelativeUrl)(source)}/${name}/index.${(0, languages_1.languageToShort)(language)}`,
                },
        ];
        const newExtensionOptions = {
            type: 'bundle',
            path: { app: 'dist/app.js', api: 'dist/api.js' },
            entries,
            host: extensionOptions.host,
            hidden: extensionOptions.hidden,
        };
        const newExtensionManifest = {
            ...extensionManifest,
            name: constants_1.EXTENSION_NAME_REGEX.test(extensionName) ? extensionName : `directus-extension-${extensionName}`,
            keywords: ['directus', 'directus-extension', `directus-custom-bundle`],
            [constants_1.EXTENSION_PKG_KEY]: newExtensionOptions,
            devDependencies: await (0, get_extension_dev_deps_1.default)(entries.map((entry) => entry.type), getLanguageFromEntries(entries)),
        };
        await fs_extra_1.default.writeJSON(packagePath, newExtensionManifest, { spaces: indent !== null && indent !== void 0 ? indent : '\t' });
        const packageManager = (0, get_package_manager_1.default)();
        await (0, execa_1.default)(packageManager, ['install'], { cwd: extensionPath });
        spinner.succeed(chalk_1.default.bold('Done'));
    }
}
exports.default = add;
function getLanguageFromEntries(entries) {
    const languages = new Set();
    for (const entry of entries) {
        if ((0, utils_1.isTypeIn)(entry, constants_1.HYBRID_EXTENSION_TYPES)) {
            const languageApp = (0, languages_1.getLanguageFromPath)(entry.source.app);
            const languageApi = (0, languages_1.getLanguageFromPath)(entry.source.api);
            if (!(0, languages_1.isLanguage)(languageApp)) {
                (0, logger_1.log)(`App language ${chalk_1.default.bold(languageApp)} is not supported.`, 'error');
                process.exit(1);
            }
            if (!(0, languages_1.isLanguage)(languageApi)) {
                (0, logger_1.log)(`API language ${chalk_1.default.bold(languageApi)} is not supported.`, 'error');
                process.exit(1);
            }
            languages.add(languageApp);
            languages.add(languageApi);
        }
        else {
            const language = (0, languages_1.getLanguageFromPath)(entry.source);
            if (!(0, languages_1.isLanguage)(language)) {
                (0, logger_1.log)(`Language ${chalk_1.default.bold(language)} is not supported.`, 'error');
                process.exit(1);
            }
            languages.add(language);
        }
    }
    return Array.from(languages);
}
