"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.GoogleCloudStorage = void 0;
const drive_1 = require("@directus/drive");
const path_1 = __importDefault(require("path"));
const normalize_path_1 = __importDefault(require("normalize-path"));
const lodash_1 = require("lodash");
function handleError(err, path) {
    switch (err.code) {
        case 401:
            return new drive_1.AuthorizationRequired(err, path);
        case 403:
            return new drive_1.PermissionMissing(err, path);
        case 404:
            return new drive_1.FileNotFound(err, path);
        case 'ENOENT':
            return new drive_1.WrongKeyPath(err, path);
        default:
            return new drive_1.UnknownException(err, String(err.code), path);
    }
}
class GoogleCloudStorage extends drive_1.Storage {
    constructor(config) {
        super();
        // This is necessary as only credentials are in snake_case, not camelCase. Ref #8601
        if (config.credentials) {
            config.credentials = (0, lodash_1.mapKeys)(config.credentials, (_value, key) => (0, lodash_1.snakeCase)(key));
        }
        this.$config = config;
        const GCSStorage = require('@google-cloud/storage').Storage;
        this.$driver = new GCSStorage(config);
        this.$bucket = this.$driver.bucket(config.bucket);
        this.$root = config.root ? (0, normalize_path_1.default)(config.root).replace(/^\//, '') : '';
    }
    /**
     * Prefixes the given filePath with the storage root location
     */
    _fullPath(filePath) {
        return (0, normalize_path_1.default)(path_1.default.join(this.$root, filePath));
    }
    _file(filePath) {
        return this.$bucket.file(this._fullPath(filePath));
    }
    /**
     * Copy a file to a location.
     */
    async copy(src, dest) {
        const srcFile = this._file(src);
        const destFile = this._file(dest);
        try {
            const result = await srcFile.copy(destFile);
            return { raw: result };
        }
        catch (e) {
            throw handleError(e, src);
        }
    }
    /**
     * Delete existing file.
     */
    async delete(location) {
        try {
            const result = await this._file(location).delete();
            return { raw: result, wasDeleted: true };
        }
        catch (e) {
            const error = handleError(e, location);
            if (error instanceof drive_1.FileNotFound) {
                return { raw: undefined, wasDeleted: false };
            }
            throw error;
        }
    }
    /**
     * Returns the driver.
     */
    driver() {
        return this.$driver;
    }
    /**
     * Determines if a file or folder already exists.
     */
    async exists(location) {
        try {
            const result = await this._file(location).exists();
            return { exists: result[0], raw: result };
        }
        catch (e) {
            throw handleError(e, location);
        }
    }
    /**
     * Returns the file contents.
     */
    async get(location, encoding = 'utf-8') {
        try {
            const result = await this._file(location).download();
            return { content: result[0].toString(encoding), raw: result };
        }
        catch (e) {
            throw handleError(e, location);
        }
    }
    /**
     * Returns the file contents as Buffer.
     */
    async getBuffer(location) {
        try {
            const result = await this._file(location).download();
            return { content: result[0], raw: result };
        }
        catch (e) {
            throw handleError(e, location);
        }
    }
    /**
     * Returns signed url for an existing file.
     */
    async getSignedUrl(location, options = {}) {
        const { expiry = 900 } = options;
        try {
            const result = await this._file(location).getSignedUrl({
                action: 'read',
                expires: Date.now() + expiry * 1000,
            });
            return { signedUrl: result[0], raw: result };
        }
        catch (e) {
            throw handleError(e, location);
        }
    }
    /**
     * Returns file's size and modification date.
     */
    async getStat(location) {
        try {
            const result = await this._file(location).getMetadata();
            return {
                size: Number(result[0].size),
                modified: new Date(result[0].updated),
                raw: result,
            };
        }
        catch (e) {
            throw handleError(e, location);
        }
    }
    /**
     * Returns the stream for the given file.
     */
    getStream(location, range) {
        return this._file(location).createReadStream({ start: range === null || range === void 0 ? void 0 : range.start, end: range === null || range === void 0 ? void 0 : range.end });
    }
    /**
     * Returns URL for a given location. Note this method doesn't
     * validates the existence of file or it's visibility
     * status.
     */
    getUrl(location) {
        return `https://storage.googleapis.com/${this.$bucket.name}/${this._fullPath(location)}`;
    }
    /**
     * Move file to a new location.
     */
    async move(src, dest) {
        const srcFile = this._file(src);
        const destFile = this._file(dest);
        try {
            const result = await srcFile.move(destFile);
            return { raw: result };
        }
        catch (e) {
            throw handleError(e, src);
        }
    }
    /**
     * Creates a new file.
     * This method will create missing directories on the fly.
     */
    async put(location, content) {
        const file = this._file(location);
        try {
            if ((0, drive_1.isReadableStream)(content)) {
                const destStream = file.createWriteStream({ resumable: false });
                await (0, drive_1.pipeline)(content, destStream);
                return { raw: undefined };
            }
            const result = await file.save(content, { resumable: false });
            return { raw: result };
        }
        catch (e) {
            throw handleError(e, location);
        }
    }
    /**
     * Iterate over all files in the bucket.
     */
    async *flatList(prefix = '') {
        prefix = this._fullPath(prefix);
        let nextQuery = {
            prefix,
            autoPaginate: false,
            maxResults: 1000,
        };
        do {
            try {
                const result = (await this.$bucket.getFiles(nextQuery));
                nextQuery = result[1];
                for (const file of result[0]) {
                    yield {
                        raw: file.metadata,
                        path: file.name.substring(this.$root.length),
                    };
                }
            }
            catch (e) {
                throw handleError(e, prefix);
            }
        } while (nextQuery);
    }
}
exports.GoogleCloudStorage = GoogleCloudStorage;
