"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.AzureBlobWebServicesStorage = void 0;
const drive_1 = require("@directus/drive");
const storage_blob_1 = require("@azure/storage-blob");
const path_1 = __importDefault(require("path"));
const stream_1 = require("stream");
const normalize_path_1 = __importDefault(require("normalize-path"));
function handleError(err, path) {
    return new drive_1.UnknownException(err, err.name, path);
}
class AzureBlobWebServicesStorage extends drive_1.Storage {
    constructor(config) {
        var _a;
        super();
        this.$signedCredentials = new storage_blob_1.StorageSharedKeyCredential(config.accountName, config.accountKey);
        this.$client = new storage_blob_1.BlobServiceClient((_a = config.endpoint) !== null && _a !== void 0 ? _a : `https://${config.accountName}.blob.core.windows.net`, this.$signedCredentials);
        this.$containerClient = this.$client.getContainerClient(config.containerName);
        this.$root = config.root ? (0, normalize_path_1.default)(config.root).replace(/^\//, '') : '';
    }
    /**
     * Prefixes the given filePath with the storage root location
     */
    _fullPath(filePath) {
        return (0, normalize_path_1.default)(path_1.default.join(this.$root, filePath));
    }
    async copy(src, dest) {
        src = this._fullPath(src);
        dest = this._fullPath(dest);
        try {
            const source = this.$containerClient.getBlockBlobClient(src);
            const target = this.$containerClient.getBlockBlobClient(dest);
            const poller = await target.beginCopyFromURL(source.url);
            const result = await poller.pollUntilDone();
            return { raw: result };
        }
        catch (e) {
            throw handleError(e, src);
        }
    }
    async delete(location) {
        location = this._fullPath(location);
        try {
            const result = await this.$containerClient.getBlockBlobClient(location).deleteIfExists();
            return { raw: result, wasDeleted: result.succeeded };
        }
        catch (e) {
            throw handleError(e, location);
        }
    }
    driver() {
        return this.$client;
    }
    async exists(location) {
        location = this._fullPath(location);
        try {
            const result = await this.$containerClient.getBlockBlobClient(location).exists();
            return { exists: result, raw: result };
        }
        catch (e) {
            throw handleError(e, location);
        }
    }
    async get(location, encoding = 'utf-8') {
        try {
            const bufferResult = await this.getBuffer(location);
            return {
                content: bufferResult.content.toString(encoding),
                raw: bufferResult.raw,
            };
        }
        catch (e) {
            throw new drive_1.FileNotFound(e, location);
        }
    }
    async getBuffer(location) {
        location = this._fullPath(location);
        try {
            const client = this.$containerClient.getBlobClient(location);
            return { content: await client.downloadToBuffer(), raw: client };
        }
        catch (e) {
            throw handleError(e, location);
        }
    }
    async getSignedUrl(location, options = {}) {
        location = this._fullPath(location);
        const { expiry = 900 } = options;
        try {
            const client = this.$containerClient.getBlobClient(location);
            const blobSAS = (0, storage_blob_1.generateBlobSASQueryParameters)({
                containerName: this.$containerClient.containerName,
                blobName: location,
                permissions: storage_blob_1.ContainerSASPermissions.parse('racwdl'),
                startsOn: new Date(),
                expiresOn: new Date(new Date().valueOf() + expiry),
            }, this.$signedCredentials).toString();
            const sasUrl = client.url + '?' + blobSAS;
            return { signedUrl: sasUrl, raw: client };
        }
        catch (e) {
            throw handleError(e, location);
        }
    }
    async getStat(location) {
        location = this._fullPath(location);
        try {
            const props = await this.$containerClient.getBlobClient(location).getProperties();
            return {
                size: props.contentLength,
                modified: props.lastModified,
                raw: props,
            };
        }
        catch (e) {
            throw handleError(e, location);
        }
    }
    getStream(location, range) {
        location = this._fullPath(location);
        const intermediateStream = new stream_1.PassThrough({ highWaterMark: 1 });
        const stream = this.$containerClient
            .getBlobClient(location)
            .download(range === null || range === void 0 ? void 0 : range.start, (range === null || range === void 0 ? void 0 : range.end) ? range.end - (range.start || 0) : undefined);
        try {
            stream
                .then((result) => result.readableStreamBody)
                .then((stream) => {
                if (!stream) {
                    throw handleError(new Error('Blobclient stream was not available'), location);
                }
                stream.pipe(intermediateStream);
            })
                .catch((error) => {
                intermediateStream.emit('error', error);
            });
        }
        catch (error) {
            intermediateStream.emit('error', error);
        }
        return intermediateStream;
    }
    getUrl(location) {
        location = this._fullPath(location);
        return this.$containerClient.getBlobClient(location).url;
    }
    async move(src, dest) {
        src = this._fullPath(src);
        dest = this._fullPath(dest);
        const source = this.$containerClient.getBlockBlobClient(src);
        const target = this.$containerClient.getBlockBlobClient(dest);
        const poller = await target.beginCopyFromURL(source.url);
        const result = await poller.pollUntilDone();
        await source.deleteIfExists();
        return { raw: result };
    }
    async put(location, content, type) {
        location = this._fullPath(location);
        const blockBlobClient = this.$containerClient.getBlockBlobClient(location);
        try {
            if ((0, drive_1.isReadableStream)(content)) {
                const result = await blockBlobClient.uploadStream(content, undefined, undefined, {
                    blobHTTPHeaders: { blobContentType: type !== null && type !== void 0 ? type : 'application/octet-stream' },
                });
                return { raw: result };
            }
            const result = await blockBlobClient.upload(content, content.length);
            return { raw: result };
        }
        catch (e) {
            throw handleError(e, location);
        }
    }
    async *flatList(prefix = '') {
        prefix = this._fullPath(prefix);
        try {
            const blobs = this.$containerClient.listBlobsFlat({
                prefix,
            });
            for await (const blob of blobs) {
                yield {
                    raw: blob,
                    path: blob.name.substring(this.$root.length),
                };
            }
        }
        catch (e) {
            throw handleError(e, prefix);
        }
    }
}
exports.AzureBlobWebServicesStorage = AzureBlobWebServicesStorage;
